
// WebSocket connection state management
let socket = null;           // Active WebSocket instance
let currentWS = null;        // WebSocket URL we're connected to
let subscribedTopic = null;  // Topic the socket subscribed to
let reconnectTimer = null;   // Timer for reconnection backoff
let reconnectDelayMs = 1000; // Exponential backoff delay (1s -> 30s max)
let enabled = true;          // Global enable/disable flag from storage
let keepAliveTimer = null;   // Interval timer for keepalive pings

// Per-tab connection status tracking
// Maps tabId -> {status: 'connected'|'disconnected'|'disabled', topic: string}
const tabState = new Map();

// Server health probe cache to avoid excessive health checks
// Structure: {url: string, ok: boolean, ts: timestamp}
let serverProbeCache = { url: null, ok: false, ts: 0 };

// Mutex to serialize connection attempts and prevent race conditions
let connectionMutex = Promise.resolve();

// Global topic constant - all clients subscribe to the same topic for mass broadcasting
const GLOBAL_TOPIC = "ai-mass-form-filler";

// Icon paths for different connection states
const ICONS = {
  connected: {
    16: "icons/icon-green-16.png",
    32: "icons/icon-green-32.png",
    48: "icons/icon-green-48.png",
    128: "icons/icon-green-128.png",
  },
  disconnected: {
    16: "icons/icon-gray-16.png",
    32: "icons/icon-gray-32.png",
    48: "icons/icon-gray-48.png",
    128: "icons/icon-gray-128.png",
  }
};

/**
 * Canonicalize URL to topic name for subscription.
 * Returns global topic constant so all tabs receive all broadcasts.
 * 
 * @param {string} _u - Page URL (unused but kept for API compatibility)
 * @returns {string} The global topic string
 */
function canonTopic(_u) {
  return GLOBAL_TOPIC;
}

async function getActiveTab() {
  const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
  return tabs && tabs[0] ? tabs[0] : null;
}

async function getActiveTopic() {
  const t = await getActiveTab();
  if (!t || !t.url) return "";
  return canonTopic(t.url);
}

function setIcon(tabId, state) {
  const path = ICONS[state === "connected" ? "connected" : "disconnected"];
  chrome.action.setIcon({ tabId, path });
  if (state === "connected") {
    chrome.action.setBadgeBackgroundColor({ tabId, color: "#00C853" });
    chrome.action.setBadgeText({ tabId, text: " " }); // green dot
  } else if (state === "disabled") {
    chrome.action.setBadgeBackgroundColor({ tabId, color: "#B00020" });
    chrome.action.setBadgeText({ tabId, text: "OFF" });
  } else {
    chrome.action.setBadgeText({ tabId, text: "" });
  }
}

function setTabStatus(tabId, status, topic) {
  tabState.set(tabId, { status, topic });
  setIcon(tabId, status);
}

function resetBackoff() { reconnectDelayMs = 1000; }
function scheduleReconnect() {
  if (reconnectTimer) clearTimeout(reconnectTimer);
  // Add random jitter to prevent thundering herd
  const jitter = Math.random() * 1000; // 0-1 second random jitter
  reconnectTimer = setTimeout(ensureSocket, reconnectDelayMs + jitter);
  reconnectDelayMs = Math.min(reconnectDelayMs * 2, 30000);
}

async function probeServer(wsUrl) {
  try {
    const now = Date.now();
    if (serverProbeCache.url === wsUrl && serverProbeCache.ok && (now - serverProbeCache.ts) < 10000) {
      return true;
    }
    const url = new URL(wsUrl);
    url.protocol = url.protocol === "wss:" ? "https:" : "http:";
    url.pathname = "/healthz";
    url.search = "";
    const resp = await fetch(url.toString(), { method: "GET", cache: "no-store" });
    const ok = resp.ok;
    serverProbeCache = { url: wsUrl, ok, ts: now };
    return ok;
  } catch (err) {
    serverProbeCache = { url: wsUrl, ok: false, ts: Date.now() };
    return false;
  }
}

function stopKeepAlive() {
  if (keepAliveTimer) {
    clearInterval(keepAliveTimer);
    keepAliveTimer = null;
  }
}

async function closeSocket(updateIconForTabId = null) {
  stopKeepAlive();
  try { if (socket) socket.onmessage = socket.onopen = socket.onclose = socket.onerror = null; } catch {}
  try { if (socket && socket.readyState === WebSocket.OPEN) socket.close(); } catch {}
  socket = null;
  if (updateIconForTabId != null) {
    const st = tabState.get(updateIconForTabId);
    const state = (st && st.status === "disabled") ? "disabled" : "disconnected";
    setTabStatus(updateIconForTabId, state, st ? st.topic : undefined);
  }
}

/**
 * Ensure WebSocket connection is established for the active tab.
 * 
 * This is the main connection management function that:
 * 1. Serializes all connection attempts with a mutex to prevent races
 * 2. Checks if service is enabled and server is healthy
 * 3. Reuses existing connection if already subscribed to correct topic
 * 4. Establishes new connection if needed
 * 5. Sets up keepalive ping loop
 * 6. Updates icon state for the active tab
 * 
 * Called on:
 * - Tab activation/update
 * - URL navigation
 * - Settings changes
 * - Manual reconnect requests
 * - Reconnection backoff timer
 */
async function ensureSocket() {
  // Serialize all connection attempts to prevent races
  connectionMutex = connectionMutex
    .then(async () => {
      // Load settings from storage
      const { enabled: storeEnabled = true, ws = "ws://localhost:8765/ws", alias = "" } = await chrome.storage.local.get({ enabled: true, ws: "ws://localhost:8765/ws", alias: "" });
      enabled = !!storeEnabled;

      // Get current active tab
      const activeTab = await getActiveTab();
      if (!activeTab || !activeTab.id) return;
      const targetTabId = activeTab.id;

  // Determine topic for current tab
  const topic = await getActiveTopic();
  const targetTopic = topic;

  // If service is disabled, close socket and update icon
  if (!enabled) {
    await closeSocket(targetTabId);
    setTabStatus(targetTabId, "disabled", targetTopic || "n/a");
    return;
  }

  // If no valid topic, mark disconnected and retry later
  if (!targetTopic) {
    setTabStatus(targetTabId, "disconnected", "n/a");
    scheduleReconnect();
    return;
  }

  // Check if server is healthy before attempting connection
  const serverReady = await probeServer(ws);
  if (!serverReady) {
    setTabStatus(targetTabId, "disconnected", targetTopic);
    scheduleReconnect();
    return;
  }

  // If already connected and subscribed to the active topic, just reflect state
  const isOpen = socket && socket.readyState === WebSocket.OPEN;
  if (currentWS === ws && subscribedTopic === targetTopic && isOpen) {
    setTabStatus(targetTabId, "connected", targetTopic);
    resetBackoff();
    return;
  }

  // (Re)connect for the current tab/topic; use closure-stable tab and topic
  await closeSocket(null);
  currentWS = ws;
  subscribedTopic = targetTopic;

  setTabStatus(targetTabId, "disconnected", targetTopic); // show gray while connecting

  let wsock;
  try {
    wsock = new WebSocket(currentWS);
  } catch (e) {
    scheduleReconnect();
    return;
  }
  socket = wsock;

  socket.onopen = () => {
    try {
      socket.send(JSON.stringify({ type: "subscribe", topic: subscribedTopic, alias: alias || "" }));
      stopKeepAlive();
      keepAliveTimer = setInterval(() => {
        if (!socket || socket.readyState !== WebSocket.OPEN) {
          stopKeepAlive();
          scheduleReconnect();
          return;
        }
        try {
          socket.send(JSON.stringify({ type: "ping", topic: subscribedTopic, at: Date.now() }));
        } catch (err) {
          stopKeepAlive();
          scheduleReconnect();
        }
      }, 25000);
      // Still on same tab/topic?
      chrome.tabs.query({ active: true, currentWindow: true }).then(tabs => {
        const now = tabs && tabs[0];
        const nowTopic = now && now.url ? canonTopic(now.url) : "";
        const sameContext = now && now.id === targetTabId && nowTopic === targetTopic;
        setTabStatus(sameContext ? targetTabId : (now ? now.id : targetTabId), "connected", sameContext ? targetTopic : nowTopic || targetTopic);
      });
      resetBackoff();
    } catch (e) {
      setTabStatus(targetTabId, "disconnected", targetTopic);
      scheduleReconnect();
    }
  };

  /**
   * Handle incoming messages from WebSocket server.
   * 
   * When a broadcast message arrives:
   * 1. Parse JSON payload
   * 2. Deliver to all frames in the current active tab
   * 3. Each frame's content script will execute the action plan
   * 
   * This enables support for:
   * - Multi-frame/iframe forms
   * - Shadow DOM elements across frames
   * - Dynamic content loaded after initial page load
   */
  socket.onmessage = async (ev) => {
    let msg;
    try { msg = JSON.parse(ev.data); } catch { return; }
    const payload = msg && msg.payload;
    if (!payload) return;
    
    // Deliver to all frames of the *current* active tab only
    const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
    const now = tabs && tabs[0];
    if (!now) return;
    
    try {
      // Get all frames in the active tab
      const frames = await chrome.webNavigation.getAllFrames({ tabId: now.id });
      if (frames && frames.length) {
        // Send message to each frame individually
        for (const f of frames) {
          try { 
            await chrome.tabs.sendMessage(now.id, { type: "REMOTE_PLAN", payload }, { frameId: f.frameId }); 
          } catch {}
        }
      } else {
        // Fallback: send to main frame only
        await chrome.tabs.sendMessage(now.id, { type: "REMOTE_PLAN", payload });
      }
    } catch {
      // Final fallback if frame enumeration fails
      try { await chrome.tabs.sendMessage(now.id, { type: "REMOTE_PLAN", payload }); } catch {}
    }
  };

  const onDown = async () => {
    stopKeepAlive();
    const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
    const now = tabs && tabs[0];
    const stId = now ? now.id : targetTabId;
    setTabStatus(stId, enabled ? "disconnected" : "disabled", now && now.url ? canonTopic(now.url) : targetTopic);
    scheduleReconnect();
  };

  socket.onclose = onDown;
  socket.onerror = () => {
    stopKeepAlive();
    try { socket.close(); } catch {}
  };
    })
    .catch(err => {
      console.error('[Background] ensureSocket error:', err);
    });
  
  return connectionMutex;
}

// Messaging for popup
chrome.runtime.onMessage.addListener((msg, _sender, sendResponse) => {
  (async () => {
    if (msg?.type === "GET_STATUS") {
      const { enabled: storeEnabled = true, ws = "ws://localhost:8765/ws", alias = "" } = await chrome.storage.local.get({ enabled: true, ws: "ws://localhost:8765/ws", alias: "" });
      const activeTab = await getActiveTab();
      const topic = await getActiveTopic();
      const isOpen = socket && socket.readyState === WebSocket.OPEN;
      const connected = !!storeEnabled && isOpen && subscribedTopic === topic && !!topic;
      const tabId = activeTab ? activeTab.id : -1;
      const ts = tabState.get(tabId);
      if (!ts) tabState.set(tabId, { status: connected ? "connected" : (storeEnabled ? "disconnected" : "disabled"), topic: topic || "n/a" });
      sendResponse({
        enabled: !!storeEnabled,
        ws,
        alias,
        topic: topic || "n/a",
        socketState: isOpen ? "OPEN" : (socket ? (socket.readyState === WebSocket.CONNECTING ? "CONNECTING" : socket.readyState === WebSocket.CLOSING ? "CLOSING" : "CLOSED") : "CLOSED"),
        connected
      });
    } else if (msg?.type === "SET_ENABLED") {
      await chrome.storage.local.set({ enabled: !!msg.enabled });
      ensureSocket();
      sendResponse({ ok: true });
    } else if (msg?.type === "RECONNECT_NOW") {
      resetBackoff();
      const activeTab = await getActiveTab();
      await closeSocket(activeTab ? activeTab.id : null);
      ensureSocket();
      sendResponse({ ok: true });
    }
  })();
  return true;
});

// React to tab/runtime changes
chrome.runtime.onInstalled.addListener(ensureSocket);
chrome.runtime.onStartup.addListener(ensureSocket);
chrome.tabs.onActivated.addListener(ensureSocket);
chrome.tabs.onUpdated.addListener((_id, info, _tab) => {
  if (info.status === "complete" || info.url) ensureSocket();
});
// Catch SPA route changes
chrome.webNavigation.onHistoryStateUpdated.addListener((_d) => ensureSocket());
chrome.webNavigation.onCommitted.addListener((_d) => ensureSocket());
chrome.storage.onChanged.addListener((changes, area) => {
  if (area === "local" && (changes.ws || changes.enabled)) ensureSocket();
});
