
(function(){
  // Debug logging flag (loaded lazily from extension settings)
  let DEBUG = false;
  chrome.storage && chrome.storage.local && chrome.storage.local.get({ debug: false }, s => { DEBUG = !!s.debug; });

  /**
   * Debug logger - only logs when DEBUG flag is enabled
   */
  function log(...args){ if (DEBUG) console.log("[RFF]", ...args); }
  
  /**
   * Normalize string for case-insensitive matching
   * Collapses whitespace and converts to lowercase
   */
  function norm(s){ return (s||"").replace(/\s+/g," ").trim().toLowerCase(); }
  
  // DOM query result cache to avoid repeated expensive queries
  // Maps selector -> array of matching elements
  const queryCache = new Map();
  const CACHE_MAX_AGE = 5000; // Cache validity: 5 seconds
  let cacheTimestamp = 0;
  
  /**
   * Extract normalized ID signature for fuzzy ID matching.
   * Removes separators and digits to match fields with dynamic IDs
   * (e.g., "user-name-123" matches "username-456")
   */
  function idSignature(s){
    return norm(s).replace(/[-_]/g, "").replace(/\d+/g, "");
  }

  /**
   * Deep query across document and shadow DOM boundaries.
   * 
   * Standard querySelectorAll() doesn't pierce shadow DOM, so this function:
   * 1. Searches the provided root node
   * 2. Walks the DOM tree to find shadow roots
   * 3. Recursively searches each shadow root
   * 4. Caches results for 5 seconds to avoid expensive re-queries
   * 
   * This enables the extension to find fields inside web components
   * and custom elements with shadow DOM encapsulation.
   * 
   * @param {Node} root - Root node to search from (usually document)
   * @param {string} selector - CSS selector to match
   * @returns {Array<Element>} All matching elements across shadow boundaries
   */
  function deepQueryAll(root, selector) {
    const now = Date.now();
    const key = selector;
    
    // Return cached results if fresh (performance optimization)
    if (now - cacheTimestamp < CACHE_MAX_AGE && queryCache.has(key)) {
      log('cache hit for', selector);
      return queryCache.get(key);
    }
    
    const out = [];
    const visited = new Set();
    
    /**
     * Recursive walker that traverses shadow DOM boundaries
     */
    function walk(node, depth = 0) {
      // Prevent infinite recursion and revisiting nodes
      if (!node || depth > 10 || visited.has(node)) return;
      visited.add(node);
      
      // Query the current node
      try {
        const els = node.querySelectorAll(selector);
        els && els.forEach ? els.forEach(e => out.push(e)) : null;
      } catch(e){}
      
      // Walk tree to find shadow roots
      const treeWalker = document.createTreeWalker(node, NodeFilter.SHOW_ELEMENT, null);
      let cur = treeWalker.currentNode;
      while (cur) {
        // Recursively search shadow roots
        if (cur.shadowRoot) walk(cur.shadowRoot, depth + 1);
        cur = treeWalker.nextNode();
      }
    }
    walk(root);
    
    // Cache the results for future queries
    queryCache.set(key, out);
    return out;
  }

  function findByLabelText(text) {
    if (!text) return null;
    const want = norm(text);
    const labels = deepQueryAll(document, "label");
    for (const l of labels) {
      const lt = norm(l.innerText || l.textContent || "");
      if (!lt) continue;
      if (lt === want || lt.includes(want)) {
        if (l.htmlFor) {
          const c = document.getElementById(l.htmlFor);
          if (c) return c;
        }
        const c = l.querySelector("input, textarea, select");
        if (c) return c;
      }
    }
    return null;
  }

  function cssEscapeCompat(s) {
    if (typeof CSS !== "undefined" && CSS.escape) return CSS.escape(s);
    return String(s).replace(/\\/g, "\\\\").replace(/"/g, '\\"');
  }

  function findByAriaLabel(label) {
    if (!label) return null;
    const want = norm(label);
    const els = deepQueryAll(document, "[aria-label], [aria-labelledby]");
    for (const el of els) {
      const al = el.getAttribute("aria-label");
      if (al && norm(al) === want) return el;
      const alby = el.getAttribute("aria-labelledby");
      if (alby) {
        const refIds = alby.split(/\s+/).filter(Boolean);
        for (const rid of refIds) {
          const ref = document.getElementById(rid) || deepQueryAll(document, `#${cssEscapeCompat(rid)}`)[0];
          if (ref && norm(ref.textContent||"") === want) return el;
        }
      }
    }
    return null;
  }

  function findByLooseId(id) {
    if (!id) return null;
    const sig = idSignature(id);
    if (!sig) return null;
    const candidates = deepQueryAll(document, "[id]");
    for (const el of candidates) {
      const cid = el.id || "";
      if (!cid) continue;
      const csig = idSignature(cid);
      if (!csig) continue;
      if (csig === sig || csig.startsWith(sig) || sig.startsWith(csig)) return el;
    }
    return null;
  }

  /**
   * Resolve field metadata to actual DOM element.
   * 
   * Uses multiple fallback strategies in priority order:
   * 1. HTML ID (direct getElementById + shadow DOM search + fuzzy match)
   * 2. Name attribute
   * 3. Associated label text
   * 4. ARIA label/labelledby
   * 5. Placeholder text
   * 
   * This multi-strategy approach handles various form patterns:
   * - Fields with/without IDs
   * - Dynamic IDs (with fuzzy matching)
   * - Label associations (explicit <label for=""> or implicit nesting)
   * - Accessibility attributes
   * - Fields identified only by placeholder
   * 
   * @param {Object} field - Field metadata from agent discovery
   * @returns {Element|null} The resolved DOM element, or null if not found
   */
  function resolveElement(field) {
    // 1) ID-based resolution (exact + shadow DOM + fuzzy)
    if (field.html_id) {
      const el = document.getElementById(field.html_id) || deepQueryAll(document, `#${cssEscapeCompat(field.html_id)}`)[0] || findByLooseId(field.html_id);
      if (el) return el;
    }
    
    // 2) Name attribute resolution
    if (field.name) {
      const byName = deepQueryAll(document, `[name="${cssEscapeCompat(field.name)}"]`);
      if (byName[0]) return byName[0];
    }
    
    // 3) Label text association
    if (field.label) {
      const el = findByLabelText(field.label);
      if (el) return el;
    }
    
    // 4) ARIA label resolution
    if (field.label) {
      const el = findByAriaLabel(field.label);
      if (el) return el;
    }
    
    // 5) Placeholder text matching
    if (field.placeholder) {
      const candidates = deepQueryAll(document, "input[placeholder],textarea[placeholder]");
      const want = norm(field.placeholder);
      for (const el of candidates) {
        if (norm(el.getAttribute("placeholder")) === want) return el;
      }
    }
    
    return null;
  }

  async function typeValue(el, val) {
    try { el.focus(); } catch {}
    el.dispatchEvent(new Event("focus", { bubbles: true }));
    el.value = "";
    el.dispatchEvent(new Event("input", { bubbles: true }));
    el.value = val || "";
    el.dispatchEvent(new Event("input", { bubbles: true }));
    el.dispatchEvent(new Event("change", { bubbles: true }));
    el.dispatchEvent(new Event("blur", { bubbles: true }));
  }

  /**
   * Apply action plan to the current page.
   * 
   * Executes each action sequentially:
   * 1. Resolve field metadata to DOM element using multi-strategy resolution
   * 2. Scroll element into view for visibility
   * 3. Add green outline for visual feedback
   * 4. Execute action based on type (fill, select, check, radio, click)
   * 5. Dispatch appropriate DOM events to trigger form validation
   * 6. Short delay between actions for stability
   * 7. Remove outline
   * 
   * Action types:
   * - fill_element: Fill text input/textarea with value
   * - set_checked: Set checkbox state (checked/unchecked)
   * - choose_radio: Select radio button
   * - select_option: Select dropdown option by label/value/index
   * - click_element: Generic element click
   * 
   * Event dispatching (focus/input/change/blur) ensures that:
   * - React/Vue components detect changes
   * - Form validation runs
   * - onChange handlers fire
   * 
   * @param {Object} payload - Broadcast payload with fields and actions arrays
   */
  async function applyActions(payload) {
    try {
      const { fields = [], actions = [] } = payload || {};
      // Build field lookup map for O(1) access
      const fmap = Object.fromEntries(fields.map(f => [f.field_id, f]));
      log("received actions", actions.length, "fields", fields.length);

      // Execute actions sequentially
      for (const act of actions) {
        const kind = act.kind || act.type;
        const fid = act.field_id;
        const f = fmap[fid];
        if (!f) { log("no field for", fid); continue; }
        
        // Resolve field to DOM element
        const el = resolveElement(f);
        if (!el) { log("no element resolved for", f); continue; }

        // Scroll into view and highlight
        el.scrollIntoView({ block: "center", behavior: "smooth" });
        const prevOutline = el.style.outline;
        el.style.outline = "3px solid #00e676";

        // Execute action based on type
        if (kind === "fill_element") {
          await typeValue(el, act.text || "");
        } else if (kind === "set_checked") {
          // Checkbox/toggle action
          if ("checked" in el) {
            el.dispatchEvent(new Event("focus", { bubbles: true }));
            el.checked = !!act.checked;
            el.dispatchEvent(new Event("change", { bubbles: true }));
            el.dispatchEvent(new Event("blur", { bubbles: true }));
          }
        } else if (kind === "choose_radio") {
          // Radio button selection
          el.click();
        } else if (kind === "select_option") {
          // Dropdown selection
          const sel = el.tagName && el.tagName.toLowerCase() === "select" ? el : null;
          if (sel) {
            const by = act.match_by, opt = act.option;
            let chosen = null;
            const n = (x)=> (x==null?"":String(x)).trim();

            // Match by label text (most common)
            if (by === "label") {
              for (const o of sel.options) {
                if (norm(o.label) === norm(n(opt))) { chosen = o.value; break; }
              }
            } 
            // Match by option value attribute
            else if (by === "value") {
              for (const o of sel.options) {
                if (o.value === n(opt)) { chosen = o.value; break; }
              }
            } 
            // Match by numeric index
            else if (by === "index") {
              const idx = Number(opt);
              if (!Number.isNaN(idx) && sel.options[idx]) chosen = sel.options[idx].value;
            }
            
            // Apply selection and dispatch events
            if (chosen !== null) {
              el.dispatchEvent(new Event("focus", { bubbles: true }));
              sel.value = chosen;
              sel.dispatchEvent(new Event("input", { bubbles: true }));
              sel.dispatchEvent(new Event("change", { bubbles: true }));
              el.dispatchEvent(new Event("blur", { bubbles: true }));
            }
          }
        } else if (kind === "click_element") {
          // Generic click action
          el.click();
        }

        // Short delay for stability and visual feedback
        await new Promise(r => setTimeout(r, 60));
        el.style.outline = prevOutline || "";
      }
    } catch (e) {
      console.warn("[RFF] applyActions failed:", e);
    }
  }

  function runWhenReady(cb) {
    if (document.readyState === "complete" || document.readyState === "interactive") cb();
    else document.addEventListener("DOMContentLoaded", cb, { once: true });
  }

  /**
   * Message listener for broadcast plans from background script.
   * 
   * When a REMOTE_PLAN message arrives:
   * 1. Clear DOM query cache (page may have changed)
   * 2. Wait for DOM ready state
   * 3. Execute action plan against current page
   * 
   * The cache clear ensures fresh element resolution after dynamic content loads.
   */
  chrome.runtime.onMessage.addListener((msg, _sender, _sendResponse) => {
    if (msg?.type === "REMOTE_PLAN") {
      // Clear cache on new plan to avoid stale element references
      queryCache.clear();
      cacheTimestamp = Date.now();
      
      // Wait for DOM ready before executing actions
      runWhenReady(() => applyActions(msg.payload));
    }
  });

  log("content script loaded in frame:", window.location.href);
})();
