#!/usr/bin/env bash
set -Eeuo pipefail

# Insta360 MediaSDK & CameraSDK automated installer for Linux
# This script installs the system dependencies, registers the MediaSDK package,
# and compiles the stitching demo described in README.md.

SCRIPT_DIR=$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)
REPO_ROOT=$(realpath "${SCRIPT_DIR}/../.." 2>/dev/null || echo "${SCRIPT_DIR}/../..")
MODELS_SDK_DIR="${REPO_ROOT}/models/insta360"
DEFAULT_SDK_ROOT="${SCRIPT_DIR}/libMediaSDK-dev-3.0.5.1-20250618_195946-amd64"
SDK_ROOT="${INSTA360_SDK_BASE:-$DEFAULT_SDK_ROOT}"
DEFAULT_DEB="${SDK_ROOT}/libMediaSDK-dev-3.0.5.1-20250618_195946-amd64.deb"
EXAMPLE_DIR="${SDK_ROOT}/example"
DEFAULT_CAMERA_ROOT_SCRIPT="${SCRIPT_DIR}/CameraSDK-20250418_145834-2.0.2-Linux"
DEFAULT_CAMERA_ROOT_MODELS="${MODELS_SDK_DIR}/CameraSDK-20250418_145834-2.0.2-Linux"
MEDIA_SDK_SO="/usr/lib/libMediaSDK.so"
OUTPUT_BINARY="${EXAMPLE_DIR}/main"

APT_PACKAGES=(
  vulkan-tools
  mesa-utils
  libopencv-dev
  build-essential
  pkg-config
)

info()    { printf '\033[1;34m==> %s\033[0m\n' "$*"; }
success() {
  local msg="$*"
  printf '\n\033[1;32m%s\033[0m\n' "============================================================"
  printf '\033[1;32m|| %s ||\033[0m\n' "$msg"
  printf '\033[1;32m%s\033[0m\n\n' "============================================================"
}
warn()    { printf '\033[1;33m[warning]\033[0m %s\n' "$*"; }
error()   { printf '\033[1;31m[error]\033[0m %s\n' "$*" >&2; exit 1; }
require_cmd() {
  command -v "$1" >/dev/null 2>&1 || error "Comando richiesto non trovato: $1"
}

detect_deb_package() {
  if [[ -n "${MEDIA_SDK_DEB:-}" ]]; then
    printf '%s\n' "$MEDIA_SDK_DEB"
    return 0
  fi

  if [[ -f "$DEFAULT_DEB" ]]; then
    printf '%s\n' "$DEFAULT_DEB"
    return 0
  fi

  local search_dirs=()
  [[ -d "$SCRIPT_DIR" ]] && search_dirs+=("$SCRIPT_DIR")
  [[ -d "$SDK_ROOT" ]] && search_dirs+=("$SDK_ROOT")
  [[ -d "$MODELS_SDK_DIR" ]] && search_dirs+=("$MODELS_SDK_DIR")

  local found
  for dir in "${search_dirs[@]}"; do
    found=$(find "$dir" -maxdepth 3 -type f -name 'libMediaSDK-dev-*.deb' -print -quit 2>/dev/null || true)
    if [[ -n "$found" ]]; then
      printf '%s\n' "$found"
      return 0
    fi
  done

  printf '%s\n' ""
}

detect_camera_sdk_root() {
  if [[ -n "${INSTA360_CAMERA_SDK_BASE:-}" ]]; then
    printf '%s\n' "$INSTA360_CAMERA_SDK_BASE"
    return 0
  fi

  for candidate in "$DEFAULT_CAMERA_ROOT_SCRIPT" "$DEFAULT_CAMERA_ROOT_MODELS"; do
    if [[ -d "$candidate" ]]; then
      printf '%s\n' "$candidate"
      return 0
    fi
  done

  local search_dirs=()
  [[ -d "$SCRIPT_DIR" ]] && search_dirs+=("$SCRIPT_DIR")
  [[ -d "$SDK_ROOT" ]] && search_dirs+=("$SDK_ROOT")
  [[ -d "$MODELS_SDK_DIR" ]] && search_dirs+=("$MODELS_SDK_DIR")

  local found
  for dir in "${search_dirs[@]}"; do
    found=$(find "$dir" -maxdepth 3 -type d -name 'CameraSDK-*-Linux' -print -quit 2>/dev/null || true)
    if [[ -n "$found" ]]; then
      printf '%s\n' "$found"
      return 0
    fi
  done

  printf '%s\n' ""
}

main() {
  info "Verifica strumenti di base"
  require_cmd dpkg
  require_cmd apt-get

  if [[ $EUID -ne 0 ]]; then
    require_cmd sudo
    SUDO="sudo"
  else
    SUDO=""
  fi

  info "Installazione dipendenze di sistema (${APT_PACKAGES[*]})"
  $SUDO apt-get update
  $SUDO apt-get install -y --no-install-recommends "${APT_PACKAGES[@]}"

  local deb_file
  deb_file=$(detect_deb_package)

  if [[ -d "$SDK_ROOT" ]]; then
    SDK_ROOT=$(realpath "$SDK_ROOT" 2>/dev/null || echo "$SDK_ROOT")
  else
    warn "Directory MediaSDK non trovata in $SDK_ROOT"
    cat <<EOF
Per proseguire:
  - Estrai la cartella libMediaSDK-dev-* accanto a questo script
  - Oppure esporta INSTA360_SDK_BASE=/percorso/libMediaSDK-dev-*
EOF
    return 0
  fi
  EXAMPLE_DIR="${SDK_ROOT}/example"
  OUTPUT_BINARY="${EXAMPLE_DIR}/main"

  if [[ -z "$deb_file" ]]; then
    if [[ -f "$MEDIA_SDK_SO" ]]; then
      info "MediaSDK già presente in $MEDIA_SDK_SO, salto l'installazione del pacchetto .deb"
    else
      cat <<EOF
[error] Pacchetto MediaSDK non trovato. Scarica/estrai "libMediaSDK-dev-*.deb" e riprova.
  Opzioni:
    - Posiziona il file .deb in ${SDK_ROOT} o in ${MODELS_SDK_DIR}
    - Oppure esporta MEDIA_SDK_DEB="/percorso/al/file.deb" prima di eseguire lo script
EOF
      exit 1
    fi
  else
    info "Installazione MediaSDK da $deb_file"
    if ! $SUDO dpkg -i "$deb_file"; then
      warn "dpkg ha segnalato dipendenze mancanti, provo a risolvere"
      $SUDO apt-get install -f -y
      $SUDO dpkg -i "$deb_file"
    fi

    if command -v ldconfig >/dev/null 2>&1; then
      $SUDO ldconfig
    fi
  fi

  if [[ -f "$MEDIA_SDK_SO" ]]; then
    success "Installazione di libMediaSDK completata: $MEDIA_SDK_SO"
  else
    warn "libMediaSDK.so non trovata in /usr/lib. Verifica manualmente l'installazione."
  fi

  local camera_root
  camera_root=$(detect_camera_sdk_root)

  local camera_include="${INSTA360_CAMERA_SDK_INCLUDE:-${camera_root:+${camera_root}/include}}"
  local camera_lib="${INSTA360_CAMERA_SDK_LIB:-${camera_root:+${camera_root}/lib}}"
  local use_camera_sdk=false

  if [[ -n "$camera_include" && -d "$camera_include" ]]; then
    camera_include=$(realpath "$camera_include" 2>/dev/null || echo "$camera_include")
  else
    camera_include=""
  fi

  if [[ -n "$camera_lib" && -d "$camera_lib" ]]; then
    camera_lib=$(realpath "$camera_lib" 2>/dev/null || echo "$camera_lib")
  else
    camera_lib=""
  fi

  local ld_hint="export LD_LIBRARY_PATH=\"/usr/lib\""
  if [[ -n "$camera_include" && -n "$camera_lib" ]]; then
    use_camera_sdk=true
    info "CameraSDK trovato: $camera_root"
    ld_hint="export LD_LIBRARY_PATH=\"$camera_lib:/usr/lib\""
  else
    warn "CameraSDK non trovato. Procedo comunque con la compilazione usando solo la MediaSDK."
  fi

  require_cmd g++

  info "Compilazione demo di stitching"
  pushd "$EXAMPLE_DIR" >/dev/null
  if [[ "$use_camera_sdk" == true ]]; then
    g++ -std=c++11 \
        -I../include \
        -I"$camera_include" \
        -I/usr/include/opencv4 \
        -L"$camera_lib" \
        -L/usr/lib \
        main.cc \
        -lopencv_core -lopencv_imgcodecs -lopencv_imgproc -lopencv_highgui -lopencv_videoio \
        -lCameraSDK -lMediaSDK \
        -o main
  else
    g++ -std=c++11 \
        -I../include \
        -I/usr/include/opencv4 \
        -L/usr/lib \
        main.cc \
        -lopencv_core -lopencv_imgcodecs -lopencv_imgproc -lopencv_highgui -lopencv_videoio \
        -lMediaSDK \
        -o main
  fi
  popd >/dev/null

  if [[ -x "$OUTPUT_BINARY" ]]; then
    info "Compilazione completata: $OUTPUT_BINARY"
    cat <<EOF

Per eseguire la demo:
  cd "$EXAMPLE_DIR"
  $ld_hint
  ./main -inputs /percorso/video.insv -image_sequence_dir output_frames -image_type jpg -output_size 11520x5760 -stitch_type template
  # Modifica i parametri sopra in base alle tue necessità (vedi README).
EOF
  else
    warn "Compilazione completata ma l'eseguibile non è stato trovato: $OUTPUT_BINARY"
  fi
}

main "$@"
